#include <stdafx.h>
#include <vd2/system/vdtypes.h>
#include <vd2/system/vdstl.h>
#include <vd2/system/cpuaccel.h>
#include <at/atcore/checksum.h>
#include <test.h>
#include <stdexcept>

#include <windows.h>
#include <bcrypt.h>
#pragma comment(lib, "bcrypt.lib")

#if VD_CPU_X86 || VD_CPU_X64
#include <intrin.h>
#endif

namespace {
	class CPUCoreLock {
	public:
		CPUCoreLock() {
			DWORD_PTR processMask = ~(DWORD_PTR)0;
			DWORD_PTR systemMask;
			GetProcessAffinityMask(GetCurrentProcess(), &processMask, &systemMask);

			mOldAffinityMask = SetThreadAffinityMask(GetCurrentThread(), processMask ^ (processMask & (processMask - 1)));
		}

		~CPUCoreLock() {
			if (mOldAffinityMask)
				SetThreadAffinityMask(GetCurrentThread(), mOldAffinityMask);
		}

		DWORD_PTR mOldAffinityMask;
	};

	consteval ATChecksumSHA256 operator""_atsha256(const char *s, size_t len) {
		if (len != 64)
			throw std::invalid_argument("invalid SHA256 checksum length");

		ATChecksumSHA256 checksum {};

		for(size_t i=0; i<64; ++i) {
			char c = s[i];
			int digit = 0;

			if (c >= '0' && c <= '9')
				digit = (int)(c - '0');
			else if (c >= 'A' && c <= 'F')
				digit = (int)(c - 'A') + 10;
			else if (c >= 'a' && c <= 'f')
				digit = (int)(c - 'a') + 10;
			else
				throw std::invalid_argument("invalid character in SHA256 digest");

			if (i & 1)
				checksum.mDigest[i >> 1] += digit;
			else
				checksum.mDigest[i >> 1] = digit << 4;
		}

		return checksum;
	}
}

DEFINE_TEST(Core_Checksum) {
	const auto test = [](const char *name) {
		ATChecksumSHA256 c, d;
	
		c = ATComputeChecksumSHA256("", 0);
		d = "E3B0C44298FC1C149AFBF4C8996FB92427AE41E4649B934CA495991B7852B855"_atsha256;
		TEST_ASSERT(c == d);

		c = ATComputeChecksumSHA256("abc", 3);
		d = "BA7816BF8F01CFEA414140DE5DAE2223B00361A396177A9CB410FF61F20015AD"_atsha256;
	
		c = ATComputeChecksumSHA256("abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq", 3);
		d = "248D6A61D20638B8E5C026930C3E6039A33CE45964FF2167F6ECEDD419DB06C1"_atsha256;

		// Generated from Python:
		// import hashlib;
		// print('\n'.join([hashlib.sha256(''.join([chr(65 + ((x*x) % 26)) for x in range(0,len)]).encode('utf-8')).hexdigest().upper() for len in range(1,129)]))

		static const ATChecksumSHA256 kChecksums[] = {
			"559AEAD08264D5795D3909718CDD05ABD49572E84FE55590EEF31A88A08FDFFD"_atsha256,
			"38164FBD17603D73F696B8B4D72664D735BB6A7C88577687FD2AE33FD6964153"_atsha256,
			"9A125F9934CBB32249A8B4644A01324A95E9226046AACE680120240BE4492393"_atsha256,
			"535BF0B8E53C47B1A635E33085A156A71B513A41033F624E69B14FEEBB39A07A"_atsha256,
			"71B5B4335DD75E9F4770A5C2F4F268B4F50B38330E1EAC39818347C8FEC1685F"_atsha256,
			"CC38229CCF0048FBFDF1A5DE2BD76501B14EF96536FBAA692F11201F8C588933"_atsha256,
			"18BBA9D94BB92B025AAD11DD161A6C44E0B9DC965CC577C1EC5FF54B7A6FAE00"_atsha256,
			"5AFCB313A287ED684C9C61C82846AA8C488373680820CD55DFBEF93AA9D5CCDE"_atsha256,
			"2E1BB92E62C97035CF5A7F051125962B33967943B72E59058E323C62168AAAFC"_atsha256,
			"1705819D200B152DFD040508BE26D489B1D07813995D0248BCBB081AD9F3A002"_atsha256,
			"BBD5B8AE43CDB286934A94EE49E7FDABA7EE6D076E4EB6DFE5A24D72B3185855"_atsha256,
			"59F3DD8C87B9701328FC645100BB8ED9C116D0EB382F431D33D79E1B14166353"_atsha256,
			"408B56369A03F0469A3D9FDAE174387B28F70CE0D97CB60789772F26ED40942A"_atsha256,
			"DE9A32B685A4B2C702725A7BE4954B86B2F2495528359688A5A4F9AF7F301053"_atsha256,
			"0DF1DBC0B7BBDDA5B85989389B58649EE1ECB7C77A475D55754A8161D5A6A6CC"_atsha256,
			"126FC0D70145B557769D5DA996EB5C0766510FE17C026DD08681505BED93861A"_atsha256,
			"F73AAC59E76178CD797B4900D97D1F904B69C530C12AE2569B1B28D0971CDA53"_atsha256,
			"A4D373E1463075341E74EF1016E1564B009800BD92673F9C9A085D564A6D5046"_atsha256,
			"A8CA0129A99523FD2779C43F6927675668E3DC46C046A2958E63D314BC225BF0"_atsha256,
			"205BE32B572054D995A578958B9CEBB825861FAE343932AE81FDCA2879C6526B"_atsha256,
			"D98E8DCB79C7FF9EF19A6434AEDC8DA4940C0FFB8C4A0F656D15869921AC5D2D"_atsha256,
			"F0D80C638022C95A6327894C726C2F6E04C2F8C398FDA4BF7D51F6DC5388808A"_atsha256,
			"85210257DFDF302CB1D3039BB66447D46411ECCCB7C0A08D11DD1AB825D8BE7E"_atsha256,
			"C0A829795C26E05CD57AB870AD61C061D48BD6794AE0B3BC970132A97D45BEE9"_atsha256,
			"1DF3B2B33F9D9F9DEE7F1835F0872C116144E728084C5A5EAB1B687CEF71C95D"_atsha256,
			"58087E2340CD73126359A84D5BC9E77D3A496143993F3F5C6707A93B619C57F6"_atsha256,
			"F5DC25098FC090CFC7B7477EB786A61328A9DB17B719C7FAD0198ECDB02505B8"_atsha256,
			"B05E68EFECB65297C8FD88B4E13A27055AF8F4C20B895519C65FB5503AD97840"_atsha256,
			"CFA5F97641265B98344BBACEF06A78CF881184A4B015AF62C8AF68C29FC67ACB"_atsha256,
			"BBE228CF1B63C64C60ADCECB15D63B68EB3F9E6384CAA491091F22F76E6E9D80"_atsha256,
			"802C91233D4E9F7DA6F5901A14F2400F01FE57AE8DE9D1DAE03EA4C9D4F54C7F"_atsha256,
			"8EA22CD04F399EC3535014E48BB5AF3810A44548C3493E487A2CC0C514FA99C2"_atsha256,
			"9F75C9EA8CDAEE8D70BB1CBE39A898981975AB62C20C0377253EB54F2291CB92"_atsha256,
			"330EB1F3BE0D32B889F0865A96CE80DA4D6768D556DAC2CD2E35667FC1C34497"_atsha256,
			"6738460228BF73626F5FFDC9137F4AB90C0D05883C4733B46E13AAF3B2CE5846"_atsha256,
			"9426A3C7414BF6BC8AAF13C1003F11DC1F4182950137B2B80DC58B3978A7F1BE"_atsha256,
			"BDD16AECFC40211CC8FD514E74CA354D71692A6C0FF81F194D641EDF4F43AB5C"_atsha256,
			"F28C0A8BCF18520A3C3C37CB65D77511A5E3E3E1D12DDC7E1420FAD01477F2E9"_atsha256,
			"ECCA70B0BA5F3BC7441A5909AF0414152144E953F25AB5BE07330F6D834B98EC"_atsha256,
			"E1ED054F3815FCD97DD168E993FE763E6089268827F184BF4A0D23D0FA1C955D"_atsha256,
			"D6705BDD862D76D1E142BB305820F17C315459C7A830D34840582BFE21682D5D"_atsha256,
			"E63DCE7A692DB97537A2866BB2CAF2C5B37A8A0758A3D83F4443B9B43A4754DB"_atsha256,
			"1CAE5D8B0EEEAABDD36846F7A2C3AE398C830182B25ADC3A451BFBA7F23C32D6"_atsha256,
			"57F77A3228DE9466C1203976547C7049B6CA06F9CCE9783A8F97EE6AC8A9C774"_atsha256,
			"C6D1C0CDDE941081681947D6E9E431B680ADAF39529C859849AAC49883DC1CAF"_atsha256,
			"5EADCB2E8D53F1B931BC65D9C71F32F192FA82FB39C63BB23B3B6B3B620E12C1"_atsha256,
			"D1594BE8DF7962738971429968F3995428B0CF8752FAA35F96258CBF8BD0E5F3"_atsha256,
			"555B536B1243AD7CC5307B1213D2A4EA7E93483C0D40119255A98FE317E876FF"_atsha256,
			"DCF069ED349747571C95629D32402B24C52BF8C1030D04D1F1040401BA862A5D"_atsha256,
			"CF67135366D7E4393E9151607F5CCE5BAFCCA57E14E8E211A1954424597FCC9B"_atsha256,
			"BB16ABD76E18879EDFBD10D80C1F84FA4498CC620307B7603661EF5FF23E1F35"_atsha256,
			"5D55A1F1BD349D735C820DB50213A4D722D49C1ABC66554F001F9CE1B1DAEE5B"_atsha256,
			"DCA3A6332961B008B606809F76BD95AFDA0C5D28ED274922207C9653CB78DBF8"_atsha256,
			"1533A6BC266CDE1CBA0DE9C1B16BD24315706D010EFF78A3403EFCB30148C80B"_atsha256,
			"9063348A35BE7BA6CBD7E08360B4316C1838D923FC5C9F482CE488EEDD1B3CFB"_atsha256,
			"616189B5F1CAA0FB5ABC983D2CD1B32C057F1B7705F38918AFB2D2D167C0EB26"_atsha256,
			"964D80083F1D65DE4253C5D934EE345EDC5D162C632DE2E722DB896C18B3C987"_atsha256,
			"213D9BC53182DB88DACA38C1B64E082B843A0B68BF4C839412C514C428044DBA"_atsha256,
			"1906555B47FE1B7A890A05BEBE0EE2E2C2531ACD4D0AF50CF28C17AB6717EAF2"_atsha256,
			"441C072BA0FF05F7A8B3FB9AD95DD773670E64A93517C1CE1744F045BBF438EA"_atsha256,
			"56322D025EF638EA0CFF44A53899E22E775A12F678B041D1FDFC749C83088C0B"_atsha256,
			"6BED645A8FCF2BF41B0B6D94A3A4519857EE8C8DAE49CF9B8DD5D1E212129142"_atsha256,
			"7C053830A20BF8196946E12C36102386EF3CA329330ED9F2E651FC518CBCB2C1"_atsha256,
			"565D6AABB785A416B55C713FE98DE1123456B9D211A2B2A5FC339E394F580F6B"_atsha256,
			"B659BC38E693C82AC596CE3990D464C60B0E1412FA8E34ADDC49CAC2D969A1A2"_atsha256,
			"0C754DC62A2286D49BA515250AD7EA1B518FC12AD26EDB07F1D56DBD0ECCF337"_atsha256,
			"8F260B65E91DF71A7D881DE487FCCD6FC37040EAA4ECF5850A3FE40FEA864DCF"_atsha256,
			"C0776E5EF74E0479203FBF6FED1785E900EDB9A01BE71024356F6BDF48D2BB4C"_atsha256,
			"3832F5A9236D2A3218F0EABB58984782A6F8A2C04008CD62BE4D63B66B27D893"_atsha256,
			"660CAC4748527D48204A842A7E33CA2D4E3D7FB4677E84ED269E77CE595DAC98"_atsha256,
			"153178AED5BF91EE345C14392E11E661A84449E71AFFC8B1F797FEC757AABE08"_atsha256,
			"0F9BA55759524CE38C3973384C912C8640BD06DBF59411CA9613A7E247927987"_atsha256,
			"7412C9931AC049CD51BD06E9A230DFEACE616213614354B791D4C69CCA1D5844"_atsha256,
			"B240C3AEE6DEEB1EA6672D80F439F8DF73CD1742ABB42473F694360734BD6487"_atsha256,
			"70671BB1F624A5776437F8404E72A83F1FA1193922CF6D36A91A41C9934C255F"_atsha256,
			"47C76806E20364436DFEA0224CF4152CBCF0471A8320B0F6A189A5A6DBE1AE23"_atsha256,
			"7049521480FF6D6B691EF9849470E0E94553D3F88E68354C781348AC3ADE44DA"_atsha256,
			"F78CBD381A10CED7459B5C5B803CC7E3E9AA5174F1E260D277114360E28B6896"_atsha256,
			"068D390140D982407C280633E557C71044DF03209C5B499158D4DDB54601B5DC"_atsha256,
			"BAAAACEEF85BCB6C9AACC56048E36B41847ABC44FDACF62EAC8A5442E273A40C"_atsha256,
			"7BCEB3B34632346FCDC293C434669AE139FF1B49CB2979653DAC54621F236137"_atsha256,
			"612F7C5166720849FC5E3ED1F6BDDD8EEA57F4346FCA914E5D5871026FCF5FF5"_atsha256,
			"7B3B321ACA834CFE033D2AADFCDB51E2717B1CE403A669488292520215C72540"_atsha256,
			"48104E4A98070054A04F9968CBAC761F71A042AC19396FF33FF331E202EB8BC1"_atsha256,
			"0765DA9FAF6DA0ACE8EDBF02E0D4088EA2F8B0B1FB46027E84F726B35F222F17"_atsha256,
			"A734137109D504FDA887F0B0224B53767C3D8582B268CE11D7BC642F3478BA31"_atsha256,
			"1CAC45D423C6A82C1D0708044F883889703B10D5DB9269EAEAA2AB5107B2E34E"_atsha256,
			"8E4DFEF4C8869EB4DC3658F7368BA252130B081181F011A97B5E6921E530C6AF"_atsha256,
			"0F1024419CEC5B1F4A8DB56E26771E9F7D85A34AF0D8BA9BE974637C36F3A9F0"_atsha256,
			"E44D1B8C9CD53BBBD373E4ACF25BE64E0645D89AC38D5F1DEE1EDB53B1EF04E6"_atsha256,
			"44209941851191671641882BFA9257689D9D3409948822C73923ABF7D6D78CCF"_atsha256,
			"80F153BD28F323B0DD9C9D9197D385BB1605E740F07E232A28F113DE461D3838"_atsha256,
			"8E7C5A08570D6A52F1D7482A2AF7E95B48E6FE0996EB7B224CA2777055CECC8D"_atsha256,
			"14E1761FC4F79B6DD3A59D076BFB3BF07EFB993531B159CAD1995A945B7B63A8"_atsha256,
			"6896BB98F1CC3A76893BF3946919EB0AEDEB6AF2A06CE8140F3D331AB42C803B"_atsha256,
			"BAAB87719C87468B5106952237102F4457D51D67F2E87098E4B51F69C247915D"_atsha256,
			"A1DE7DFB0E20BC18E5B83268C3F6517E2488F0864E33F7C5D07C39E6B49088B4"_atsha256,
			"75AEB83BF5FF34E5D836BC1E941543457BF3B30284AED88D2130087635F4677F"_atsha256,
			"50B532CF390DEB05CFD1F5AA5BD196FF32A7697EBBFDF631EA7D48630D1A864F"_atsha256,
			"532AF1AB5690FA1C3B17A447666C889E407235C2FFE85DE5E2C8BE4A506AF309"_atsha256,
			"596788E6D4B82FEBF413CAFB9C9290E50CEEC21310932050618C7BBDF879EA8B"_atsha256,
			"2F24690929F89CE9B47A9E911188CCA838626D32F4A295C79DD4009E9A482666"_atsha256,
			"866E84B6B68F016FA325D06E8AECFE520D6766FC7B86A59D267B38227BD2CA36"_atsha256,
			"68D20A2B05E95DF4A20F0272C57F6356460BB2C97BBC66A2FCE23D83B1898B5D"_atsha256,
			"C1B1F0390866790CDEB1044D1F19387611EE0E3BC3620AADAAB462675E287DA6"_atsha256,
			"B113340B0EFA40CDF4B3783878AEADD2D33F6EAB690CD68C4A6DF7F018A84CB0"_atsha256,
			"DEEA74909577B7D7269E5AD2DB66336ADB6F2B0E63F257E9026E1A5B461A89B0"_atsha256,
			"6F92778C005EEDA7A6CBCBED9CA3CB613E0BC9CC10B86F0BAB5D27AB62B729A0"_atsha256,
			"E4647A0E75444DCDF236A5C454D094E3072C109A35CDBDB3C15836BF02BA0144"_atsha256,
			"328FD0A8A91D6976447C509E7553CABC6398CFD73B5ACCA449369E4E288B6A50"_atsha256,
			"D688405BB78B2A1B075CC0AE88CBD440F7A2A9240CB5C26F7BCFC2F2451492DE"_atsha256,
			"A6B60D8BDDF9BC9BF96E255A4D24EC23A895024D5D8FAF0C5F59BC1C3A5FE91B"_atsha256,
			"AB6B2DF77E0A2F872192E47C63555AB8AF55F63EA9DE6ED01A0F9110D89598B5"_atsha256,
			"1F50DBAD5C7C9F3C229A3C9BA8C7C02A4D26DC1E52F7BD53A3C8EC8DBC01EBFE"_atsha256,
			"AF7DFE2FEB1254F87D80DA31173E3D758A443FC5B6658B4F2D361E3831977C0A"_atsha256,
			"1D0D4B25CA289C18F360D6451BA89D25CC44DA9A282C4CBCFBF9C84585076EA4"_atsha256,
			"FEDBA5B43EE66CE0A62F544D241AA915270BD30399EDBEEC81B3761406D2D5DF"_atsha256,
			"25F58B94938B6324E4A67B26BDCDE406D2F631C42BCC2116CF071113178EA4C8"_atsha256,
			"2932F3F9A306ECB5F5EBEC1E220602320B09A553E1A8E0AB38AF1316B9E777E3"_atsha256,
			"77509E864E4C8C186B6FE081B2169D0BD2D1CC923180C378606F3081999E2639"_atsha256,
			"0AD2E83609BACBBEF601075ABCC88FCB99ADF1ABB53C15A29EDD846FF91D582F"_atsha256,
			"77FF5A4316438B55B3B32789CD7D6610269337EFDA07712800EFF4A559C3794C"_atsha256,
			"466799D04B7B76EECD5BBEC5F8C9E13D00019AA63189577AD16F942218EDEF12"_atsha256,
			"47C5C3E54F8F0A9EC919BCBABDED679351CAD4E8A38C23B5FE506F136691D4E0"_atsha256,
			"FF2E3929B2D706088A1031A97F94E17373139C6796047324449633435D114577"_atsha256,
			"006A74D47157E3BFFDB4D192D47F7ECBE3C8AFE806A0714712DF6581C85C0B25"_atsha256,
			"ABC2A143CB7DE0C554DEB2A1FF23DD7BEEA2A42EF71AD992CC67E90701511C5D"_atsha256,
			"5D65A47CD489C6780D4854E60F93CB0C90546C6194B5F42C14EAC513FB3CAA72"_atsha256
		};

		static_assert(vdcountof(kChecksums) == 128);

		char buf[128] {};
		for(int i=0; i<128; ++i) {
			buf[i] = (char)('A' + ((i*i) % 26));

			c = ATComputeChecksumSHA256(buf, i + 1);
			d = kChecksums[i];

			TEST_ASSERT(c == d);
		}

		AT_TEST_TRACEF("%-10s  OK", name);
	};

	ATChecksumSHA256 hash128 = "5D65A47CD489C6780D4854E60F93CB0C90546C6194B5F42C14EAC513FB3CAA72"_atsha256;
	vdblock<char> block128(128);
	for(int i=0; i<128; ++i) 
		block128[i] = (char)('A' + ((i*i) % 26));

	for(int i=0; i<=128; ++i) {
		for(int j=i; j<=128; ++j) {
			ATChecksumEngineSHA256 h;
			h.Process(block128.data(), i);
			h.Process(block128.data() + i, j - i);
			h.Process(block128.data() + j, 128 - j);
			TEST_ASSERT(h.Finalize() == hash128);
		}
	}

	long ex = CPUCheckForExtensions();

#if VD_CPU_X86 || VD_CPU_X64
	CPUEnableExtensions(ex & (CPUF_SUPPORTS_MMX | CPUF_SUPPORTS_INTEGER_SSE));
	test("Scalar");

	if (ex & CPUF_SUPPORTS_SSE2) {
		CPUEnableExtensions(ex & (CPUF_SUPPORTS_MMX | CPUF_SUPPORTS_INTEGER_SSE | CPUF_SUPPORTS_SSE2));
		test("SSE2");

		if (ex & CPUF_SUPPORTS_SSSE3) {
			CPUEnableExtensions(ex & (CPUF_SUPPORTS_MMX | CPUF_SUPPORTS_INTEGER_SSE | CPUF_SUPPORTS_SSE2 | CPUF_SUPPORTS_SSSE3));
			test("SSSE3");

			if (ex & CPUF_SUPPORTS_SHA) {
				CPUEnableExtensions(ex);
				test("SHA");
			}
		}
	}
#else
	CPUEnableExtensions(0);
	test("Scalar");

	if (ex & VDCPUF_SUPPORTS_CRYPTO) {
		CPUEnableExtensions(ex & VDCPUF_SUPPORTS_CRYPTO);
		test("Crypto");
	}
#endif

	return 0;
}

DEFINE_TEST_NONAUTO(Core_ChecksumSpeed) {
	auto test = [](const char *name, auto fn) {
		CPUCoreLock coreLock;

		vdblock<char> buf(65536);

		for(int i=0; i<65536; ++i)
			buf[i] = (char)i;

		unsigned long long tmin = ~(unsigned long long)0;
		unsigned long long tmax = 0;
		unsigned long long tsum = 0;

		for(int j=0; j<1000; ++j) {
#ifdef VD_CPU_ARM64
			volatile unsigned long long t1 = _ReadStatusReg(ARM64_PMCCNTR_EL0);
#else
			volatile unsigned long long t1 = __rdtsc();
#endif
		
			[[maybe_unused]] volatile ATChecksumSHA256 ck = fn(buf.data(), buf.size());

#ifdef VD_CPU_ARM64
			volatile unsigned long long t2 = _ReadStatusReg(ARM64_PMCCNTR_EL0) - t1;
#else
			volatile unsigned long long t2 = __rdtsc() - t1;
#endif

			if (t2 < tmin)
				tmin = t2;

			if (t2 > tmax)
				tmax = t2;

			tsum += t2;
		}

		printf("%-10s %llu %llu %llu (min %.2f cpb, avg %.2f cpb)\n", name, tmin, tmax, (tsum + 500) / 1000, (double)tmin / 65536.0, (double)tsum / 65536000.0);
	};

	long ex = CPUCheckForExtensions();

#if VD_CPU_X86 || VD_CPU_X64
	CPUEnableExtensions(ex & (CPUF_SUPPORTS_MMX | CPUF_SUPPORTS_INTEGER_SSE));
	test("Scalar", ATComputeChecksumSHA256);

	if (ex & CPUF_SUPPORTS_SSE2) {
		CPUEnableExtensions(ex & (CPUF_SUPPORTS_MMX | CPUF_SUPPORTS_INTEGER_SSE | CPUF_SUPPORTS_SSE2));
		test("SSE2", ATComputeChecksumSHA256);

		if (ex & CPUF_SUPPORTS_SSSE3) {
			CPUEnableExtensions(ex & (CPUF_SUPPORTS_MMX | CPUF_SUPPORTS_INTEGER_SSE | CPUF_SUPPORTS_SSE2 | CPUF_SUPPORTS_SSSE3));
			test("SSSE3", ATComputeChecksumSHA256);

			if (ex & CPUF_SUPPORTS_SHA) {
				CPUEnableExtensions(ex);
				test("SHA", ATComputeChecksumSHA256);
			}
		}
	}
#else
	CPUEnableExtensions(0);
	test("Scalar", ATComputeChecksumSHA256);

	if (ex & VDCPUF_SUPPORTS_CRYPTO) {
		CPUEnableExtensions(ex & VDCPUF_SUPPORTS_CRYPTO);
		test("Crypto", ATComputeChecksumSHA256);
	}
#endif
	
	CPUEnableExtensions(ex);
	return 0;
}

DEFINE_TEST_NONAUTO(Core_ChecksumSpeedSystem) {
	CPUCoreLock coreLock;

	BCRYPT_ALG_HANDLE h {};
	BCryptOpenAlgorithmProvider(&h, BCRYPT_SHA256_ALGORITHM, nullptr, 0);

	vdblock<UCHAR> buf(65536);

	for(int i=0; i<65536; ++i)
		buf[i] = (UCHAR)i;

	UCHAR digest[32];
	unsigned long long tmin = ~(unsigned long long)0;
	unsigned long long tmax = 0;
	unsigned long long tsum = 0;

	for(int j=0; j<1000; ++j) {
#ifdef VD_CPU_ARM64
		volatile unsigned long long t1 = _ReadStatusReg(ARM64_PMCCNTR_EL0);
#else
		volatile unsigned long long t1 = __rdtsc();
#endif
		
		BCRYPT_HASH_HANDLE h2;
		BCryptCreateHash(h, &h2, nullptr, 0, nullptr, 0, 0);
		BCryptHashData(h2, buf.data(), buf.size(), 0);
		BCryptFinishHash(h2, digest, 32, 0);
		BCryptDestroyHash(h2);

#ifdef VD_CPU_ARM64
		volatile unsigned long long t2 = _ReadStatusReg(ARM64_PMCCNTR_EL0) - t1;
#else
		volatile unsigned long long t2 = __rdtsc() - t1;
#endif

		if (t2 < tmin)
			tmin = t2;

		if (t2 > tmax)
			tmax = t2;

		tsum += t2;
	}

	printf("%llu %llu %llu (min %.2f cpb, avg %.2f cpb)\n", tmin, tmax, (tsum + 500) / 1000, (double)tmin / 65536.0, (double)tsum / 65536000.0);
	BCryptCloseAlgorithmProvider(h, 0);

	return 0;
}
